using System;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// This class defines standard SOAP fault codes used in VistALink protocol.
	/// Works as a "string enum".
	/// </summary>
	/// <remarks>
	/// Be careful changing underlying string values for this "enum": 
	/// they are mapped directly to VistALink messages XML representation 
	/// (used in XML serialization/deserialization). 
	/// </remarks>
	public sealed class FaultCode
	{
		/// <summary>
		/// This entry corresponds to 'VersionMisMatch' SOAP fault code. 
		/// </summary>
		public static readonly FaultCode VersionMismatch = new FaultCode( "VersionMisMatch" );

		/// <summary>
		/// This entry corresponds to 'MustUnderstand' SOAP fault code. 
		/// </summary>
		public static readonly FaultCode MustUnderstand = new FaultCode( "MustUnderstand" );

		/// <summary>
		/// This entry corresponds to 'Server' SOAP fault code. 
		/// </summary>
		public static readonly FaultCode Server = new FaultCode( "Server" );

		/// <summary>
		/// This entry corresponds to 'Client' SOAP fault code. 
		/// </summary>
		public static readonly FaultCode Client = new FaultCode( "Client" );

		private readonly string _faultCode;

		/// <summary>
		/// The only private constructor. It's called upon 
		/// static initialization of "enum" members.
		/// </summary>
		/// <param name="faultCode">Status code to assign</param>
		private FaultCode( string faultCode )
		{
			_faultCode = faultCode;
		}

		/// <summary>
		/// Override for standard ToString() method.
		/// </summary>
		/// <returns>
		///		String representation of fault code as 
		///		defined in VistALink protocol specification
		/// </returns>
		public override string ToString()
		{
			return _faultCode;
		}

		/// <summary>
		/// Parses a given string and returns appropriate fault code object.
		/// Will throw ArgumentOutOfRange exception if input string cannot be parsed.
		/// </summary>
		/// <param name="faultCodeString">
		///		String representation for fault code 
		///		as defined in VistALink protocol
		///	</param>
		/// <returns>
		///		Appropriate fault code object.
		///	</returns>
		public static FaultCode Parse( string faultCodeString ) 
		{
			if( faultCodeString == null )
				throw( new ArgumentNullException( "faultCodeString" ) );

			switch( faultCodeString )
			{
				case "VersionMisMatch":	return VersionMismatch; 
				case "MustUnderstand":	return MustUnderstand; 
				case "Client":			return Client;
				case "Server":			return Server;
				default:
					throw( new StringParseException( SR.Exceptions.StringEnumValueNotFound( "Fault code", faultCodeString ) ) );
			}
		}
	}
}
